<?php
// Exibe todos os erros para facilitar a depuração. Remova ou comente em produção.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require 'db.php'; // Sua conexão com o banco de dados

/**
 * Função refatorada para importar dados de um arquivo CSV usando Prepared Statements.
 *
 * @param string $tmp_name O caminho temporário do arquivo enviado.
 * @param string $tipo 'Receita' ou 'Despesa'.
 * @param string $origem A nova nomenclatura de origem (ex: 'DESPESAS_IMOBILIARIA').
 * @param mysqli $conexao A conexão com o banco de dados.
 */
function importar_csv_seguro($tmp_name, $tipo, $origem, $conexao) {
    // Prepara a query SQL uma única vez para maior segurança e performance
    $sql = "INSERT INTO transacoes (data_transacao, tipo, descricao, valor, forma_pagamento, entidade, origem) VALUES (?, ?, ?, ?, ?, ?, ?)";
    $stmt = $conexao->prepare($sql);

    // Se a preparação da query falhar, encerra a execução.
    if ($stmt === false) {
        die('Erro ao preparar a query: ' . htmlspecialchars($conexao->error));
    }

    $handle = fopen($tmp_name, "r");
    fgetcsv($handle); // Pula a linha do cabeçalho

    $linhas_importadas = 0;
    while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
        // Tratamento dos dados da linha do CSV
        $data_transacao_obj = DateTime::createFromFormat('d/m/Y', $data[0]);
        if (!$data_transacao_obj) continue; // Pula linha se a data for inválida

        $data_transacao = $data_transacao_obj->format('Y-m-d');
        $descricao = $data[1];
        $valor = (float)str_replace(',', '.', $data[3]);

        // Define valores padrão como null
        $forma_pagamento = null;
        $entidade = null;

        // Atribui valores com base na NOVA nomenclatura de origem (removido campo categoria)
        if ($origem == 'DESPESAS_PARTICULARES_AURY' || $origem == 'DESPESAS_IMOBILIARIA') {
            $forma_pagamento = $data[4] ?? null;
        } elseif ($origem == 'RECEITA_IMOBILIARIA') {
            $entidade = $data[2] ?? null;
            $forma_pagamento = $data[4] ?? null;
        } elseif ($origem == 'COMISSOES') {
            $entidade = $data[2] ?? null;
        }

        // Associa os parâmetros à query preparada
        $stmt->bind_param('sssdsss', $data_transacao, $tipo, $descricao, $valor, $forma_pagamento, $entidade, $origem);

        // Executa a query para a linha atual
        if ($stmt->execute()) {
            $linhas_importadas++;
        }
    }
    
    fclose($handle);
    $stmt->close();
    
    return $linhas_importadas;
}

// Lógica principal: Processa o formulário quando enviado
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    
    // Confirmação de segurança para limpar a tabela
    if (isset($_POST['confirmar_limpeza'])) {
        $conexao->query("TRUNCATE TABLE transacoes");
        echo "<p style='color:orange;font-weight:bold;'>Tabela 'transacoes' foi limpa com sucesso.</p>";
    }

    echo "<h1>Iniciando importação de dados...</h1>";

    // Mapeia os campos do formulário para os tipos e as NOVAS origens
    $arquivos_para_importar = [
        'DESPESAS_PARTICULARES_AURY' => ['tipo' => 'Despesa', 'origem' => 'DESPESAS_PARTICULARES_AURY'],
        'DESPESAS_IMOBILIARIA'       => ['tipo' => 'Despesa', 'origem' => 'DESPESAS_IMOBILIARIA'],
        'RECEITA_IMOBILIARIA'        => ['tipo' => 'Receita', 'origem' => 'RECEITA_IMOBILIARIA'],
        'COMISSOES'                  => ['tipo' => 'Receita', 'origem' => 'COMISSOES']
    ];

    foreach ($arquivos_para_importar as $campo_arquivo => $info) {
        // Verifica se o arquivo foi enviado e não houve erro
        if (isset($_FILES[$campo_arquivo]) && $_FILES[$campo_arquivo]['error'] == UPLOAD_ERR_OK) {
            $nome_original = htmlspecialchars($_FILES[$campo_arquivo]['name']);
            $tmp_name = $_FILES[$campo_arquivo]['tmp_name'];
            $linhas = importar_csv_seguro($tmp_name, $info['tipo'], $info['origem'], $conexao);
            echo "<p style='color:green;'><b>$linhas linhas</b> importadas com sucesso do arquivo: $nome_original (Origem: {$info['origem']})</p>";
        }
    }

    echo "<h2>Importação finalizada!</h2>";
    $conexao->close();
    exit(); // Termina o script para não re-exibir o formulário
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Importação de Dados Financeiros</title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif; background-color: #f4f7fa; color: #333; margin: 0; padding: 20px; display: flex; justify-content: center; align-items: center; min-height: 100vh;}
        .container { width: 100%; max-width: 600px; background: #fff; padding: 25px 40px; border-radius: 8px; box-shadow: 0 4px 15px rgba(0,0,0,0.1); }
        h1 { text-align: center; color: #2c3e50; margin-bottom: 20px;}
        .navigation { display: flex; justify-content: center; gap: 20px; margin-bottom: 30px; }
        .nav-link { padding: 10px 20px; background-color: #3498db; color: white; text-decoration: none; border-radius: 5px; transition: background-color 0.3s; }
        .nav-link:hover { background-color: #2980b9; }
        .nav-link.active { background-color: #2c3e50; }
        .form-group { margin-bottom: 20px; }
        label { display: block; margin-bottom: 8px; font-weight: bold; color: #555; }
        input[type="file"] { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; }
        .warning { background-color: #fffbe6; border: 1px solid #ffe58f; border-radius: 4px; padding: 15px; margin-top: 20px; }
        .warning label { display: flex; align-items: center; }
        .warning input[type="checkbox"] { margin-right: 10px; }
        button { width: 100%; padding: 12px; background-color: #27ae60; color: white; border: none; border-radius: 4px; font-size: 16px; font-weight: bold; cursor: pointer; transition: background-color 0.3s; }
        button:hover { background-color: #229954; }
        @media (max-width: 768px) {
            .navigation { flex-direction: column; align-items: center; }
        }
    </style>
</head>
<body>
    <div class="container">
        <div style="display: flex; align-items: center; justify-content: center; gap: 20px; margin-bottom: 20px;">
            <img src="/publico_html/assets/img/logo.png" alt="Logo" style="height: 50px; width: auto;">
            <h1>Importar Arquivos CSV</h1>
        </div>

        <div class="navigation">
            <a href="index.php" class="nav-link">Dashboard</a>
            <a href="contas.php" class="nav-link">Controle de Contas</a>
            <a href="relatorios.php" class="nav-link">Relatórios</a>
            <a href="import.php" class="nav-link active">Importar CSV</a>
        </div>

        <form action="import.php" method="post" enctype="multipart/form-data">
            
            <div class="form-group">
                <label for="despesas_particulares">Despesas Particulares (Aury)</label>
                <input type="file" name="DESPESAS_PARTICULARES_AURY" id="despesas_particulares" accept=".csv">
            </div>

            <div class="form-group">
                <label for="despesas_imobiliaria">Despesas da Imobiliária</label>
                <input type="file" name="DESPESAS_IMOBILIARIA" id="despesas_imobiliaria" accept=".csv">
            </div>

            <div class="form-group">
                <label for="receita_imobiliaria">Receita da Imobiliária</label>
                <input type="file" name="RECEITA_IMOBILIARIA" id="receita_imobiliaria" accept=".csv">
            </div>

            <div class="form-group">
                <label for="comissoes">Comissões Recebidas</label>
                <input type="file" name="COMISSOES" id="comissoes" accept=".csv">
            </div>
            
            <div class="warning">
                <label>
                    <input type="checkbox" name="confirmar_limpeza" required>
                    <strong>Atenção:</strong> Eu confirmo que todos os dados financeiros existentes serão <strong>apagados</strong> antes da nova importação.
                </label>
            </div>

            <br>
            <button type="submit">Importar Dados</button>
        </form>
    </div>
</body>
</html>
