<?php
// /gabinete/src/views/documentos/detalhes.php
if (session_status() === PHP_SESSION_NONE) { session_start(); }

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../models/Documento.php';

// Protege rota
if (empty($_SESSION['user_id'])) {
    header('Location: ' . BASE_URL . '/login.php');
    exit;
}

$documento = new Documento();
$doc = null;
$erro = '';

// Obter ID do documento
$id = intval($_GET['id'] ?? 0);

if ($id > 0) {
    try {
        // Buscar documento específico
        $pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        $stmt = $pdo->prepare("
            SELECT d.*, u.nome as nome_usuario 
            FROM documentos d 
            JOIN usuarios u ON d.id_usuario = u.id 
            WHERE d.id = :id
        ");
        $stmt->execute(['id' => $id]);
        $doc = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$doc) {
            $erro = 'Documento não encontrado.';
        }
    } catch (Exception $e) {
        $erro = 'Erro ao carregar documento: ' . $e->getMessage();
    }
} else {
    $erro = 'ID do documento inválido.';
}

$nome = $_SESSION['nome'] ?? 'Usuário';
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detalhes do Documento</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #0B1F3A;
            --secondary-color: #102B54;
            --accent-color: #3B82F6;
        }
        
        .navbar-custom {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
        }
        
        .document-preview {
            max-width: 100%;
            max-height: 500px;
            border-radius: 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .file-icon-large {
            width: 120px;
            height: 120px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 12px;
            color: white;
            font-size: 48px;
            margin: 0 auto 20px;
        }
        
        .file-icon-large.documento { background: linear-gradient(135deg, #6366f1, #8b5cf6); }
        .file-icon-large.imagem { background: linear-gradient(135deg, #10b981, #059669); }
        .file-icon-large.video { background: linear-gradient(135deg, #f59e0b, #d97706); }
        .file-icon-large.audio { background: linear-gradient(135deg, #ef4444, #dc2626); }
        
        .info-card {
            background: #f8f9fa;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .info-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 0;
            border-bottom: 1px solid #e9ecef;
        }
        
        .info-item:last-child {
            border-bottom: none;
        }
        
        .info-label {
            font-weight: 600;
            color: #495057;
        }
        
        .info-value {
            color: #6c757d;
        }
        
        .badge-categoria {
            font-size: 0.875rem;
            padding: 8px 12px;
        }
        
        .action-buttons .btn {
            margin: 0 5px;
        }
        
        .video-container {
            position: relative;
            width: 100%;
            max-width: 800px;
            margin: 0 auto;
        }
        
        .audio-player {
            width: 100%;
            max-width: 500px;
            margin: 0 auto;
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar navbar-expand-lg navbar-dark navbar-custom">
        <div class="container-fluid">
            <span class="navbar-brand">
                <i class="bi bi-file-earmark-text-fill me-2"></i>
                Detalhes do Documento
            </span>
            <div class="d-flex align-items-center text-white">
                <span class="me-3">Olá, <?= htmlspecialchars($nome) ?></span>
                <a href="lista.php" class="btn btn-outline-light btn-sm me-2">
                    <i class="bi bi-list-ul me-1"></i>Lista
                </a>
                <a href="<?= BASE_URL ?>/src/views/dashboard/" class="btn btn-outline-light btn-sm">
                    <i class="bi bi-house-fill me-1"></i>Painel
                </a>
            </div>
        </div>
    </nav>

    <div class="container py-4">
        <?php if ($erro): ?>
            <div class="alert alert-danger">
                <i class="bi bi-exclamation-triangle-fill me-2"></i>
                <?= htmlspecialchars($erro) ?>
            </div>
            <div class="text-center">
                <a href="lista.php" class="btn btn-primary">
                    <i class="bi bi-arrow-left me-1"></i>Voltar à Lista
                </a>
            </div>
        <?php elseif ($doc): ?>
            <div class="row">
                <!-- Coluna da Visualização -->
                <div class="col-lg-8">
                    <div class="card shadow">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">
                                <i class="bi bi-eye me-2"></i>
                                Visualização
                            </h5>
                            <span class="badge bg-secondary badge-categoria">
                                <?= htmlspecialchars($doc['categoria']) ?>
                            </span>
                        </div>
                        <div class="card-body text-center">
                            <?php
                            $extensao = strtolower(pathinfo($doc['nome_arquivo'], PATHINFO_EXTENSION));
                            $tipoMidia = 'documento';
                            
                            // Determinar tipo de mídia
                            if (in_array($extensao, ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'])) {
                                $tipoMidia = 'imagem';
                            } elseif (in_array($extensao, ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm'])) {
                                $tipoMidia = 'video';
                            } elseif (in_array($extensao, ['mp3', 'wav', 'ogg'])) {
                                $tipoMidia = 'audio';
                            }
                            
                            $caminhoArquivo = BASE_URL . '/' . $doc['caminho_arquivo'];
                            ?>
                            
                            <?php if ($tipoMidia === 'imagem'): ?>
                                <img src="<?= htmlspecialchars($caminhoArquivo) ?>" 
                                     alt="<?= htmlspecialchars($doc['titulo']) ?>" 
                                     class="document-preview">
                                     
                            <?php elseif ($tipoMidia === 'video'): ?>
                                <div class="video-container">
                                    <video controls class="document-preview">
                                        <source src="<?= htmlspecialchars($caminhoArquivo) ?>" type="<?= htmlspecialchars($doc['tipo_arquivo']) ?>">
                                        Seu navegador não suporta o elemento de vídeo.
                                    </video>
                                </div>
                                
                            <?php elseif ($tipoMidia === 'audio'): ?>
                                <div class="file-icon-large <?= $tipoMidia ?>">
                                    <i class="bi bi-music-note"></i>
                                </div>
                                <audio controls class="audio-player">
                                    <source src="<?= htmlspecialchars($caminhoArquivo) ?>" type="<?= htmlspecialchars($doc['tipo_arquivo']) ?>">
                                    Seu navegador não suporta o elemento de áudio.
                                </audio>
                                
                            <?php else: ?>
                                <div class="file-icon-large <?= $tipoMidia ?>">
                                    <?php
                                    $icon = match($extensao) {
                                        'pdf' => 'bi-file-earmark-pdf',
                                        'doc', 'docx' => 'bi-file-earmark-word',
                                        'txt' => 'bi-file-earmark-text',
                                        default => 'bi-file-earmark'
                                    };
                                    ?>
                                    <i class="bi <?= $icon ?>"></i>
                                </div>
                                <p class="text-muted">
                                    Clique no botão "Abrir Arquivo" para visualizar este documento.
                                </p>
                            <?php endif; ?>
                            
                            <!-- Botões de Ação -->
                            <div class="action-buttons mt-4">
                                <a href="<?= htmlspecialchars($caminhoArquivo) ?>" target="_blank" class="btn btn-primary">
                                    <i class="bi bi-box-arrow-up-right me-1"></i>Abrir Arquivo
                                </a>
                                <a href="<?= htmlspecialchars($caminhoArquivo) ?>" download class="btn btn-success">
                                    <i class="bi bi-download me-1"></i>Download
                                </a>
                                <a href="upload.php?id=<?= $doc['id'] ?>" class="btn btn-warning">
                                    <i class="bi bi-pencil me-1"></i>Editar
                                </a>
                                <button type="button" class="btn btn-danger" onclick="confirmarExclusao(<?= $doc['id'] ?>)">
                                    <i class="bi bi-trash me-1"></i>Excluir
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Coluna das Informações -->
                <div class="col-lg-4">
                    <div class="card shadow">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="bi bi-info-circle me-2"></i>
                                Informações
                            </h5>
                        </div>
                        <div class="card-body p-0">
                            <div class="info-card">
                                <div class="info-item">
                                    <span class="info-label">Título:</span>
                                    <span class="info-value"><?= htmlspecialchars($doc['titulo']) ?></span>
                                </div>
                                
                                <?php if (!empty($doc['descricao'])): ?>
                                <div class="info-item">
                                    <span class="info-label">Descrição:</span>
                                    <span class="info-value"><?= htmlspecialchars($doc['descricao']) ?></span>
                                </div>
                                <?php endif; ?>
                                
                                <div class="info-item">
                                    <span class="info-label">Nome do Arquivo:</span>
                                    <span class="info-value"><?= htmlspecialchars($doc['nome_arquivo']) ?></span>
                                </div>
                                
                                <div class="info-item">
                                    <span class="info-label">Categoria:</span>
                                    <span class="info-value">
                                        <span class="badge bg-secondary"><?= htmlspecialchars($doc['categoria']) ?></span>
                                    </span>
                                </div>
                                
                                <div class="info-item">
                                    <span class="info-label">Tipo:</span>
                                    <span class="info-value"><?= htmlspecialchars($doc['tipo_arquivo']) ?></span>
                                </div>
                                
                                <div class="info-item">
                                    <span class="info-label">Tamanho:</span>
                                    <span class="info-value"><?= formatBytes($doc['tamanho_arquivo']) ?></span>
                                </div>
                                
                                <?php if (isset($doc['largura']) && isset($doc['altura']) && $doc['largura'] && $doc['altura']): ?>
                                <div class="info-item">
                                    <span class="info-label">Dimensões:</span>
                                    <span class="info-value"><?= $doc['largura'] ?> × <?= $doc['altura'] ?> pixels</span>
                                </div>
                                <?php endif; ?>
                                
                                <div class="info-item">
                                    <span class="info-label">Enviado por:</span>
                                    <span class="info-value"><?= htmlspecialchars($doc['nome_usuario']) ?></span>
                                </div>
                                
                                <div class="info-item">
                                    <span class="info-label">Data de Upload:</span>
                                    <span class="info-value"><?= date('d/m/Y H:i', strtotime($doc['data_upload'])) ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Card de Ações Adicionais -->
                    <div class="card shadow mt-3">
                        <div class="card-header">
                            <h6 class="mb-0">
                                <i class="bi bi-gear me-2"></i>
                                Ações
                            </h6>
                        </div>
                        <div class="card-body">
                            <div class="d-grid gap-2">
                                <button class="btn btn-outline-primary btn-sm" onclick="copyLink()">
                                    <i class="bi bi-link-45deg me-1"></i>Copiar Link
                                </button>
                                <button class="btn btn-outline-info btn-sm" onclick="shareDocument()">
                                    <i class="bi bi-share me-1"></i>Compartilhar
                                </button>
                                <hr>
                                <button class="btn btn-outline-danger btn-sm" onclick="confirmDelete()">
                                    <i class="bi bi-trash me-1"></i>Excluir Documento
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Botão Voltar -->
            <div class="text-center mt-4">
                <a href="lista.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left me-1"></i>Voltar à Lista
                </a>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function copyLink() {
            const url = window.location.href;
            navigator.clipboard.writeText(url).then(() => {
                alert('Link copiado para a área de transferência!');
            }).catch(() => {
                // Fallback para navegadores mais antigos
                const textArea = document.createElement('textarea');
                textArea.value = url;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                alert('Link copiado para a área de transferência!');
            });
        }
        
        function shareDocument() {
            if (navigator.share) {
                navigator.share({
                    title: '<?= htmlspecialchars($doc['titulo'] ?? '') ?>',
                    text: 'Confira este documento: <?= htmlspecialchars($doc['titulo'] ?? '') ?>',
                    url: window.location.href
                });
            } else {
                copyLink();
            }
        }
        
        function confirmDelete() {
            if (confirm('Tem certeza que deseja excluir este documento? Esta ação não pode ser desfeita.')) {
                // Criar formulário para enviar requisição POST
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = 'excluir.php';
                
                // Adicionar campo hidden com o ID do documento
                const idField = document.createElement('input');
                idField.type = 'hidden';
                idField.name = 'id';
                idField.value = '<?= $doc['id'] ?>';
                form.appendChild(idField);
                
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        function confirmarExclusao(id) {
            confirmDelete();
        }
    </script>
</body>
</html>

<?php
function formatBytes($size, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    for ($i = 0; $size > 1024 && $i < count($units) - 1; $i++) {
        $size /= 1024;
    }
    return round($size, $precision) . ' ' . $units[$i];
}
?>