<?php
require_once __DIR__ . '/../config/config.php';

class MaterialPublicitario {
    private $pdo;
    
    public function __construct() {
        try {
            $this->pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
                DB_USER,
                DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
        } catch (PDOException $e) {
            throw new Exception("Erro na conexão: " . $e->getMessage());
        }
    }
    
    public function create($dados) {
        try {
            $sql = "INSERT INTO material_publicitario (
                        id_usuario, titulo, descricao, nome_arquivo, caminho_arquivo, 
                        tipo_arquivo, tamanho_arquivo, categoria, tipo_midia, 
                        largura, altura, duracao, thumbnail, campanha, 
                        data_evento, publico_alvo, tags, cor_predominante, 
                        formato_final, status
                    ) VALUES (
                        :id_usuario, :titulo, :descricao, :nome_arquivo, :caminho_arquivo,
                        :tipo_arquivo, :tamanho_arquivo, :categoria, :tipo_midia,
                        :largura, :altura, :duracao, :thumbnail, :campanha,
                        :data_evento, :publico_alvo, :tags, :cor_predominante,
                        :formato_final, :status
                    )";
            
            $stmt = $this->pdo->prepare($sql);
            
            // Definir valores padrão para campos opcionais
            $dados['largura'] = $dados['largura'] ?? null;
            $dados['altura'] = $dados['altura'] ?? null;
            $dados['duracao'] = $dados['duracao'] ?? null;
            $dados['thumbnail'] = $dados['thumbnail'] ?? null;
            $dados['campanha'] = $dados['campanha'] ?? null;
            $dados['data_evento'] = $dados['data_evento'] ?? null;
            $dados['publico_alvo'] = $dados['publico_alvo'] ?? null;
            $dados['tags'] = $dados['tags'] ?? null;
            $dados['cor_predominante'] = $dados['cor_predominante'] ?? null;
            $dados['formato_final'] = $dados['formato_final'] ?? null;
            $dados['status'] = $dados['status'] ?? 'rascunho';
            
            $stmt->execute($dados);
            return $this->pdo->lastInsertId();
            
        } catch (PDOException $e) {
            throw new Exception("Erro ao criar material publicitário: " . $e->getMessage());
        }
    }
    
    public function find($filtros = []) {
        try {
            $sql = "SELECT mp.*, u.nome as usuario_nome 
                    FROM material_publicitario mp 
                    LEFT JOIN usuarios u ON mp.id_usuario = u.id 
                    WHERE 1=1";
            
            $params = [];
            
            // Filtro por categoria
            if (!empty($filtros['categoria'])) {
                $sql .= " AND mp.categoria = :categoria";
                $params['categoria'] = $filtros['categoria'];
            }
            
            // Filtro por tipo de mídia
            if (!empty($filtros['tipo_midia'])) {
                $sql .= " AND mp.tipo_midia = :tipo_midia";
                $params['tipo_midia'] = $filtros['tipo_midia'];
            }
            
            // Filtro por busca (título, descrição, tags)
            if (!empty($filtros['busca'])) {
                $sql .= " AND (mp.titulo LIKE :busca OR mp.descricao LIKE :busca OR mp.tags LIKE :busca)";
                $params['busca'] = '%' . $filtros['busca'] . '%';
            }
            
            // Filtro por campanha
            if (!empty($filtros['campanha'])) {
                $sql .= " AND mp.campanha LIKE :campanha";
                $params['campanha'] = '%' . $filtros['campanha'] . '%';
            }
            
            // Filtro por status
            if (!empty($filtros['status'])) {
                $sql .= " AND mp.status = :status";
                $params['status'] = $filtros['status'];
            }
            
            $sql .= " ORDER BY mp.data_upload DESC";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } catch (PDOException $e) {
            throw new Exception("Erro ao buscar material publicitário: " . $e->getMessage());
        }
    }
    
    public function findById($id) {
        try {
            $sql = "SELECT mp.*, u.nome as usuario_nome 
                    FROM material_publicitario mp 
                    LEFT JOIN usuarios u ON mp.id_usuario = u.id 
                    WHERE mp.id = :id";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute(['id' => $id]);
            
            return $stmt->fetch(PDO::FETCH_ASSOC);
            
        } catch (PDOException $e) {
            throw new Exception("Erro ao buscar material publicitário: " . $e->getMessage());
        }
    }
    
    public function update($id, $dados) {
        try {
            $sql = "UPDATE material_publicitario SET 
                        titulo = :titulo,
                        descricao = :descricao,
                        categoria = :categoria,
                        tipo_midia = :tipo_midia,
                        campanha = :campanha,
                        data_evento = :data_evento,
                        publico_alvo = :publico_alvo,
                        tags = :tags,
                        cor_predominante = :cor_predominante,
                        formato_final = :formato_final,
                        status = :status
                    WHERE id = :id";
            
            $dados['id'] = $id;
            $stmt = $this->pdo->prepare($sql);
            
            return $stmt->execute($dados);
            
        } catch (PDOException $e) {
            throw new Exception("Erro ao atualizar material publicitário: " . $e->getMessage());
        }
    }
    
    public function delete($id) {
        try {
            $sql = "DELETE FROM material_publicitario WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            
            return $stmt->execute(['id' => $id]);
            
        } catch (PDOException $e) {
            throw new Exception("Erro ao deletar material publicitário: " . $e->getMessage());
        }
    }
    
    public function incrementarVisualizacao($id) {
        try {
            $sql = "UPDATE material_publicitario SET visualizacoes = visualizacoes + 1 WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            
            return $stmt->execute(['id' => $id]);
            
        } catch (PDOException $e) {
            throw new Exception("Erro ao incrementar visualização: " . $e->getMessage());
        }
    }
    
    public function incrementarDownload($id) {
        try {
            $sql = "UPDATE material_publicitario SET downloads = downloads + 1 WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            
            return $stmt->execute(['id' => $id]);
            
        } catch (PDOException $e) {
            throw new Exception("Erro ao incrementar download: " . $e->getMessage());
        }
    }
    
    public function getEstatisticas() {
        try {
            $sql = "SELECT 
                        categoria,
                        tipo_midia,
                        COUNT(*) as total_pecas,
                        SUM(visualizacoes) as total_visualizacoes,
                        SUM(downloads) as total_downloads,
                        ROUND(AVG(tamanho_arquivo)/1024/1024, 2) as tamanho_medio_mb
                    FROM material_publicitario 
                    GROUP BY categoria, tipo_midia
                    ORDER BY categoria, tipo_midia";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute();
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } catch (PDOException $e) {
            throw new Exception("Erro ao obter estatísticas: " . $e->getMessage());
        }
    }
}
?>