<?php
/**
 * Scheduler para envio automático de notificações WhatsApp
 * Este arquivo deve ser executado via cron job a cada 30 minutos
 * 
 * Exemplo de cron job (adicionar no crontab):
 * 0,30 * * * * /usr/bin/php /caminho/para/gabinete/src/cron/whatsapp_scheduler.php
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../services/WhatsAppService.php';
require_once __DIR__ . '/../config/log.php';

// Evita execução via browser
if (php_sapi_name() !== 'cli') {
    http_response_code(403);
    die('Este script só pode ser executado via linha de comando.');
}

class WhatsAppScheduler {
    private $whatsappService;
    private $logFile;
    
    public function __construct() {
        $this->whatsappService = new WhatsAppService();
        $this->logFile = __DIR__ . '/../logs/whatsapp_scheduler.log';
        $this->log("Iniciando WhatsApp Scheduler");
    }
    
    /**
     * Executa o processo de verificação e envio de notificações
     */
    public function executar() {
        try {
            if (!$this->whatsappService->isActive()) {
                $this->log("WhatsApp Service não está ativo. Encerrando.");
                return;
            }
            
            $compromissos = $this->whatsappService->buscarCompromissosParaNotificar();
            $this->log("Encontrados " . count($compromissos) . " compromissos para notificar");
            
            $sucessos = 0;
            $erros = 0;
            
            foreach ($compromissos as $compromisso) {
                try {
                    $this->processarCompromisso($compromisso);
                    $sucessos++;
                    $this->log("Notificação enviada com sucesso para compromisso ID: " . $compromisso['id']);
                } catch (Exception $e) {
                    $erros++;
                    $this->log("Erro ao enviar notificação para compromisso ID " . $compromisso['id'] . ": " . $e->getMessage());
                }
                
                // Pausa de 2 segundos entre envios para evitar rate limiting
                sleep(2);
            }
            
            $this->log("Processamento concluído. Sucessos: {$sucessos}, Erros: {$erros}");
            
        } catch (Exception $e) {
            $this->log("Erro crítico no scheduler: " . $e->getMessage());
        }
    }
    
    /**
     * Processa um compromisso individual
     */
    private function processarCompromisso($compromisso) {
        // Verifica se tem telefone válido
        if (empty($compromisso['telefone'])) {
            throw new Exception("Telefone não informado para o compromisso");
        }
        
        // Gera mensagem personalizada
        $mensagem = $this->whatsappService->processarTemplate($compromisso);
        
        // Adiciona informações do eleitor se disponível
        if (!empty($compromisso['nome_completo'])) {
            $saudacao = "Olá, " . $compromisso['nome_completo'] . "!\n\n";
            $mensagem = $saudacao . $mensagem;
        }
        
        // Envia a mensagem
        $enviado = $this->whatsappService->enviarMensagem(
            $compromisso['telefone'], 
            $mensagem, 
            $compromisso['id']
        );
        
        if ($enviado) {
            // Marca como notificado no banco
            $this->whatsappService->marcarComoNotificado($compromisso['id']);
        } else {
            throw new Exception("Falha no envio da mensagem WhatsApp");
        }
    }
    
    /**
     * Registra log com timestamp
     */
    private function log($mensagem) {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[{$timestamp}] {$mensagem}" . PHP_EOL;
        
        // Escreve no arquivo de log
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
        
        // Também exibe no console se executado via CLI
        if (php_sapi_name() === 'cli') {
            echo $logMessage;
        }
    }
    
    /**
     * Limpa logs antigos (mantém apenas últimos 30 dias)
     */
    public function limparLogsAntigos() {
        $arquivoLog = $this->logFile;
        
        if (!file_exists($arquivoLog)) {
            return;
        }
        
        $linhas = file($arquivoLog, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $linhasFiltradas = [];
        $dataLimite = date('Y-m-d', strtotime('-30 days'));
        
        foreach ($linhas as $linha) {
            // Extrai data da linha do log
            if (preg_match('/\[(\d{4}-\d{2}-\d{2})/', $linha, $matches)) {
                if ($matches[1] >= $dataLimite) {
                    $linhasFiltradas[] = $linha;
                }
            }
        }
        
        // Reescreve o arquivo apenas com logs recentes
        file_put_contents($arquivoLog, implode(PHP_EOL, $linhasFiltradas) . PHP_EOL);
        
        $this->log("Limpeza de logs concluída. Mantidas " . count($linhasFiltradas) . " linhas.");
    }
}

// Execução principal
if (php_sapi_name() === 'cli') {
    $scheduler = new WhatsAppScheduler();
    
    // Verifica argumentos da linha de comando
    $opcoes = getopt('', ['clean-logs', 'test']);
    
    if (isset($opcoes['clean-logs'])) {
        $scheduler->limparLogsAntigos();
    } elseif (isset($opcoes['test'])) {
        echo "Modo de teste - verificando configurações...\n";
        $whatsappService = new WhatsAppService();
        echo "WhatsApp ativo: " . ($whatsappService->isActive() ? 'Sim' : 'Não') . "\n";
        $compromissos = $whatsappService->buscarCompromissosParaNotificar();
        echo "Compromissos pendentes: " . count($compromissos) . "\n";
        foreach ($compromissos as $c) {
            echo "- ID: {$c['id']}, Título: {$c['titulo']}, Data: {$c['data_inicio']}, Telefone: {$c['telefone']}\n";
        }
    } else {
        $scheduler->executar();
    }
}
?>