<?php

require_once __DIR__ . '/../models/Employee.php';
require_once __DIR__ . '/../models/TimeRecord.php';
require_once __DIR__ . '/../models/Payroll.php';
require_once __DIR__ . '/../helpers/AuthHelper.php';

class RhReportsController {
    private $employeeModel;
    private $timeRecordModel;
    private $payrollModel;
    
    public function __construct() {
        $this->employeeModel = new Employee();
        $this->timeRecordModel = new TimeRecord();
        $this->payrollModel = new Payroll();
    }
    
    /**
     * Página principal de relatórios de RH
     */
    public function index() {
        AuthHelper::check();
        
        try {
            // Buscar dados para o relatório de RH
            $dados = [
                'totalColaboradores' => $this->getTotalColaboradores(),
                'porStatus' => $this->getColaboradoresPorStatus(),
                'porCargo' => $this->getColaboradoresPorCargo(),
                'porContrato' => $this->getColaboradoresPorTipoContrato(),
                'colaboradoresRecentes' => $this->getColaboradoresRecentes(),
                'estatisticasPonto' => $this->getEstatisticasPonto(),
                'folhaPagamento' => $this->getEstatisticasFolha()
            ];
            
            require_once __DIR__ . '/../views/rh/relatorios.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar relatórios de RH: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/employees');
            exit;
        }
    }
    
    private function getTotalColaboradores() {
        try {
            $colaboradores = $this->employeeModel->getAll();
            return count($colaboradores);
        } catch (Exception $e) {
            return 0;
        }
    }
    
    private function getColaboradoresPorStatus() {
        try {
            $colaboradores = $this->employeeModel->getAll();
            $porStatus = [];
            
            foreach ($colaboradores as $colaborador) {
                $status = $colaborador['status'] ?? 'ativo';
                $porStatus[$status] = ($porStatus[$status] ?? 0) + 1;
            }
            
            return $porStatus;
        } catch (Exception $e) {
            return [];
        }
    }
    
    private function getColaboradoresPorCargo() {
        try {
            $colaboradores = $this->employeeModel->getAll();
            $porCargo = [];
            
            foreach ($colaboradores as $colaborador) {
                $cargo = $colaborador['job_title'] ?? 'Não informado';
                $porCargo[$cargo] = ($porCargo[$cargo] ?? 0) + 1;
            }
            
            return $porCargo;
        } catch (Exception $e) {
            return [];
        }
    }
    
    private function getColaboradoresPorTipoContrato() {
        try {
            $colaboradores = $this->employeeModel->getAll();
            $porContrato = [];
            
            foreach ($colaboradores as $colaborador) {
                $contrato = $colaborador['contract_type'] ?? 'CLT';
                $porContrato[$contrato] = ($porContrato[$contrato] ?? 0) + 1;
            }
            
            return $porContrato;
        } catch (Exception $e) {
            return [];
        }
    }
    
    private function getColaboradoresRecentes() {
        try {
            $colaboradores = $this->employeeModel->getAll();
            
            // Ordenar por data de contratação (mais recentes primeiro)
            usort($colaboradores, function($a, $b) {
                return strtotime($b['hire_date'] ?? '1970-01-01') - strtotime($a['hire_date'] ?? '1970-01-01');
            });
            
            // Retornar apenas os 5 mais recentes
            return array_slice($colaboradores, 0, 5);
        } catch (Exception $e) {
            return [];
        }
    }
    
    private function getEstatisticasPonto() {
        try {
            // Estatísticas básicas de ponto do mês atual
            return [
                'total_registros' => 0,
                'horas_trabalhadas' => 0,
                'media_horas_dia' => 0
            ];
        } catch (Exception $e) {
            return [
                'total_registros' => 0,
                'horas_trabalhadas' => 0,
                'media_horas_dia' => 0
            ];
        }
    }
    
    private function getEstatisticasFolha() {
        try {
            // Estatísticas básicas de folha de pagamento
            return [
                'total_folhas' => 0,
                'valor_total' => 0,
                'media_salarial' => 0
            ];
        } catch (Exception $e) {
            return [
                'total_folhas' => 0,
                'valor_total' => 0,
                'media_salarial' => 0
            ];
        }
    }
}
?>