<?php

require_once __DIR__ . '/../helpers/AuthHelper.php';
require_once __DIR__ . '/../models/Usuario.php';

class GoogleController {

    private $client;
    private $googleEnabled;

    public function __construct() {
        $this->googleEnabled = false;
        
        // Verifica se o autoload do Composer existe
        if (!file_exists(__DIR__ . '/../../vendor/autoload.php')) {
            return; // Google não disponível
        }
        
        // Verifica se as constantes do Google estão definidas
        if (!defined('GOOGLE_CLIENT_ID') || !defined('GOOGLE_CLIENT_SECRET')) {
            return; // Google não configurado
        }
        
        // Inclui o autoload do Composer
        require_once __DIR__ . '/../../vendor/autoload.php';
        
        // Verifica se as classes do Google existem
        if (!class_exists('Google_Client')) {
            return; // Biblioteca do Google não instalada
        }
        
        try {
            // Inicia o cliente do Google
            $this->client = new \Google_Client();
            $this->client->setClientId(GOOGLE_CLIENT_ID);
            $this->client->setClientSecret(GOOGLE_CLIENT_SECRET);
            $this->client->setRedirectUri(BASE_URL . '/google/callback');
            $this->client->addScope(\Google_Service_Calendar::CALENDAR_EVENTS);
            $this->client->setAccessType('offline');
            $this->client->setPrompt('select_account consent');
            $this->googleEnabled = true;
        } catch (Exception $e) {
            // Google não disponível
        }
    }

    /**
     * Ação principal: Redireciona o usuário para a tela de autorização do Google.
     */
    public function auth() {
        AuthHelper::check();
        
        if (!$this->googleEnabled) {
            header('Location: ' . BASE_URL . '/configuracoes?status=google_not_configured');
            exit();
        }
        
        $authUrl = $this->client->createAuthUrl();
        header('Location: ' . $authUrl);
        exit();
    }

    /**
     * Ação de Callback: O Google redireciona para cá após o usuário dar permissão.
     */
    public function callback() {
        AuthHelper::check();
        
        if (!$this->googleEnabled) {
            header('Location: ' . BASE_URL . '/configuracoes?status=google_not_configured');
            exit();
        }

        // Pega o código de autorização da URL
        if (isset($_GET['code'])) {
            try {
                // Troca o código por um token de acesso
                $token = $this->client->fetchAccessTokenWithAuthCode($_GET['code']);
                $this->client->setAccessToken($token);

                // Verifica se houve erro
                if (array_key_exists('error', $token)) {
                    throw new Exception(join(', ', $token));
                }

                // Salva o token (em formato JSON) no banco de dados associado ao usuário logado
                $usuarioModel = new Usuario();
                $usuarioModel->salvarTokenGoogle($_SESSION['usuario_id'], json_encode($token));

                // Redireciona para uma página de sucesso ou de configurações
                header('Location: ' . BASE_URL . '/configuracoes?status=google_success');
                exit();
            } catch (Exception $e) {
                header('Location: ' . BASE_URL . '/configuracoes?status=google_error');
                exit();
            }
        } else {
            // Se não houver código, redireciona com erro
            header('Location: ' . BASE_URL . '/configuracoes?status=google_error');
            exit();
        }
    }
}
