<?php
require_once __DIR__ . '/../helpers/AuthHelper.php';
require_once __DIR__ . '/../models/Usuario.php';
require_once __DIR__ . '/../services/WhatsAppService.php';

class ConfiguracoesController {
    private $pdo;
    
    public function __construct() {
        try {
            $this->pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch (PDOException $e) {
            die("Erro na conexão com o banco de dados: " . $e->getMessage());
        }
    }

    public function index() {
        AuthHelper::check();
        $pageTitle = "Configurações";
        
        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/configuracoes/index.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }
    
    public function whatsapp() {
        AuthHelper::check();
        
        // Verifica se é admin
        if ($_SESSION['nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/configuracoes?status=access_denied');
            exit();
        }
        
        $pageTitle = "Configurações WhatsApp";
        
        // Carrega configurações do sistema
        $config = $this->getConfiguracoesSistema();
        
        // Carrega credenciais do usuário admin
        $credenciais = $this->getCredenciaisWhatsApp();
        
        // Carrega estatísticas
        $estatisticas = $this->getEstatisticasWhatsApp();
        
        // Carrega histórico recente
        $historico = $this->getHistoricoWhatsApp();
        
        require_once __DIR__ . '/../views/partials/header.php';
        require_once __DIR__ . '/../views/configuracoes/whatsapp.php';
        require_once __DIR__ . '/../views/partials/footer.php';
    }
    
    public function salvarWhatsApp() {
        AuthHelper::check();
        
        if ($_SESSION['nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/configuracoes/whatsapp?status=access_denied');
            exit();
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $tipo = $_POST['tipo'] ?? '';
            
            try {
                if ($tipo === 'geral') {
                    $this->salvarConfiguracoesGerais();
                } elseif ($tipo === 'twilio') {
                    $this->salvarCredenciaisTwilio();
                }
                
                header('Location: ' . BASE_URL . '/configuracoes/whatsapp?status=success');
            } catch (Exception $e) {
                header('Location: ' . BASE_URL . '/configuracoes/whatsapp?status=error');
            }
        }
        
        exit();
    }
    
    public function testarWhatsApp() {
        AuthHelper::check();
        
        if ($_SESSION['nivel_acesso'] !== 'Admin') {
            header('Location: ' . BASE_URL . '/configuracoes/whatsapp?status=access_denied');
            exit();
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $telefone = $_POST['telefone_teste'] ?? '';
            $mensagem = $_POST['mensagem_teste'] ?? '';
            
            try {
                $whatsappService = new WhatsAppService();
                $resultado = $whatsappService->enviarMensagem($telefone, $mensagem);
                
                if ($resultado) {
                    header('Location: ' . BASE_URL . '/configuracoes/whatsapp?status=test_success');
                } else {
                    header('Location: ' . BASE_URL . '/configuracoes/whatsapp?status=test_error');
                }
            } catch (Exception $e) {
                header('Location: ' . BASE_URL . '/configuracoes/whatsapp?status=test_error&error=' . urlencode($e->getMessage()));
            }
        }
        
        exit();
    }
    
    private function getConfiguracoesSistema() {
        $sql = "SELECT chave, valor FROM configuracoes_sistema WHERE chave LIKE 'whatsapp_%'";
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    }
    
    private function getCredenciaisWhatsApp() {
        $sql = "SELECT whatsapp_api_key, whatsapp_account_sid, whatsapp_phone_number FROM usuarios WHERE nivel_acesso = 'Admin' LIMIT 1";
        $stmt = $this->pdo->query($sql);
        return $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    }
    
    private function getEstatisticasWhatsApp() {
        $sql = "
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'Enviada' THEN 1 ELSE 0 END) as total_enviadas,
                SUM(CASE WHEN status IN ('Erro', 'Falhou') THEN 1 ELSE 0 END) as total_erros
            FROM notificacoes_whatsapp 
            WHERE DATE(data_criacao) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
        ";
        $stmt = $this->pdo->query($sql);
        $stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $stats['taxa_sucesso'] = $stats['total'] > 0 ? round(($stats['total_enviadas'] / $stats['total']) * 100, 1) : 0;
        
        return $stats;
    }
    
    private function getHistoricoWhatsApp() {
        $sql = "
            SELECT n.*, a.titulo as titulo_compromisso
            FROM notificacoes_whatsapp n
            LEFT JOIN agendas a ON n.id_agenda = a.id
            ORDER BY n.data_criacao DESC
            LIMIT 50
        ";
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    private function salvarConfiguracoesGerais() {
        $configuracoes = [
            'whatsapp_ativo' => isset($_POST['whatsapp_ativo']) ? '1' : '0',
            'whatsapp_antecedencia_horas' => $_POST['antecedencia_horas'] ?? '2',
            'whatsapp_api_provider' => $_POST['api_provider'] ?? 'twilio',
            'whatsapp_horario_inicio' => $_POST['horario_inicio'] ?? '08:00',
            'whatsapp_horario_fim' => $_POST['horario_fim'] ?? '20:00',
            'whatsapp_template_compromisso' => $_POST['template_mensagem'] ?? 'Lembrete: {titulo} em {data} às {hora}'
        ];
        
        foreach ($configuracoes as $chave => $valor) {
            $sql = "INSERT INTO configuracoes_sistema (chave, valor) VALUES (?, ?) ON DUPLICATE KEY UPDATE valor = VALUES(valor)";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$chave, $valor]);
        }
    }
    
    private function salvarCredenciaisTwilio() {
        $sql = "UPDATE usuarios SET whatsapp_account_sid = ?, whatsapp_api_key = ?, whatsapp_phone_number = ? WHERE nivel_acesso = 'Admin'";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            $_POST['twilio_account_sid'] ?? '',
            $_POST['twilio_auth_token'] ?? '',
            $_POST['twilio_phone_number'] ?? ''
        ]);
    }
}
