<?php

class Employee {
    private $pdo;
    
    public function __construct() {
        try {
            $this->pdo = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
                DB_USER,
                DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
        } catch (PDOException $e) {
            throw new Exception('Erro na conexão com o banco de dados: ' . $e->getMessage());
        }
    }
    
    /**
     * Buscar todos os colaboradores
     */
    public function getAll() {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM employees WHERE status = 'ativo' ORDER BY full_name");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            throw new Exception('Erro ao buscar colaboradores: ' . $e->getMessage());
        }
    }
    
    /**
     * Buscar colaborador por ID
     */
    public function findById($id) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM employees WHERE id = ?");
            $stmt->execute([$id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            throw new Exception('Erro ao buscar colaborador: ' . $e->getMessage());
        }
    }
    
    /**
     * Criar novo colaborador
     */
    public function create($data) {
        try {
            $sql = "INSERT INTO employees (
                full_name, cpf, rg, birth_date, address, city, state, postal_code, phone_number,
                job_title, salary, hire_date, contract_type, cost_center, department, work_location,
                latitude, longitude, bank_name, agency_number, account_number, status, created_at
            ) VALUES (
                :full_name, :cpf, :rg, :birth_date, :address, :city, :state, :postal_code, :phone_number,
                :job_title, :salary, :hire_date, :contract_type, :cost_center, :department, :work_location,
                :latitude, :longitude, :bank_name, :agency_number, :account_number, :status, NOW()
            )";
            
            $stmt = $this->pdo->prepare($sql);
            $result = $stmt->execute($data);
            
            if ($result) {
                return $this->pdo->lastInsertId();
            }
            
            return false;
        } catch (PDOException $e) {
            throw new Exception('Erro ao criar colaborador: ' . $e->getMessage());
        }
    }
    
    /**
     * Atualizar colaborador
     */
    public function update($id, $data) {
        try {
            $sql = "UPDATE employees SET 
                full_name = :full_name,
                cpf = :cpf,
                rg = :rg,
                birth_date = :birth_date,
                address = :address,
                city = :city,
                state = :state,
                postal_code = :postal_code,
                phone_number = :phone_number,
                job_title = :job_title,
                salary = :salary,
                hire_date = :hire_date,
                contract_type = :contract_type,
                cost_center = :cost_center,
                department = :department,
                work_location = :work_location,
                latitude = :latitude,
                longitude = :longitude,
                bank_name = :bank_name,
                agency_number = :agency_number,
                account_number = :account_number,
                status = :status,
                updated_at = NOW()
                WHERE id = :id";
            
            $data['id'] = $id;
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($data);
        } catch (PDOException $e) {
            throw new Exception('Erro ao atualizar colaborador: ' . $e->getMessage());
        }
    }
    
    /**
     * Deletar colaborador
     */
    public function delete($id) {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM employees WHERE id = ?");
            return $stmt->execute([$id]);
        } catch (PDOException $e) {
            throw new Exception('Erro ao deletar colaborador: ' . $e->getMessage());
        }
    }
    
    /**
     * Validar CPF
     */
    public function validateCpf($cpf) {
        // Remove caracteres não numéricos
        $cpf = preg_replace('/[^0-9]/', '', $cpf);
        
        // Verifica se tem 11 dígitos
        if (strlen($cpf) != 11) {
            return false;
        }
        
        // Verifica se todos os dígitos são iguais
        if (preg_match('/^(\d)\1{10}$/', $cpf)) {
            return false;
        }
        
        // Calcula o primeiro dígito verificador
        $sum = 0;
        for ($i = 0; $i < 9; $i++) {
            $sum += intval($cpf[$i]) * (10 - $i);
        }
        $remainder = $sum % 11;
        $digit1 = ($remainder < 2) ? 0 : 11 - $remainder;
        
        // Verifica o primeiro dígito
        if (intval($cpf[9]) != $digit1) {
            return false;
        }
        
        // Calcula o segundo dígito verificador
        $sum = 0;
        for ($i = 0; $i < 10; $i++) {
            $sum += intval($cpf[$i]) * (11 - $i);
        }
        $remainder = $sum % 11;
        $digit2 = ($remainder < 2) ? 0 : 11 - $remainder;
        
        // Verifica o segundo dígito
        return intval($cpf[10]) == $digit2;
    }
    
    /**
     * Verificar se CPF já existe
     */
    public function cpfExists($cpf, $excludeId = null) {
        try {
            $sql = "SELECT COUNT(*) FROM employees WHERE cpf = ?";
            $params = [$cpf];
            
            if ($excludeId) {
                $sql .= " AND id != ?";
                $params[] = $excludeId;
            }
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchColumn() > 0;
        } catch (PDOException $e) {
            throw new Exception('Erro ao verificar CPF: ' . $e->getMessage());
        }
    }
    
    /**
     * Contar colaboradores ativos
     */
    public function countActive() {
        try {
            $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM employees WHERE status = 'ativo'");
            $stmt->execute();
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            throw new Exception('Erro ao contar colaboradores: ' . $e->getMessage());
        }
    }
    
    /**
     * Buscar colaboradores com filtros
     */
    public function search($filters = []) {
        try {
            $sql = "SELECT * FROM employees WHERE 1=1";
            $params = [];
            
            if (!empty($filters['name'])) {
                $sql .= " AND full_name LIKE ?";
                $params[] = '%' . $filters['name'] . '%';
            }
            
            if (!empty($filters['cpf'])) {
                $sql .= " AND cpf LIKE ?";
                $params[] = '%' . $filters['cpf'] . '%';
            }
            
            if (!empty($filters['job_title'])) {
                $sql .= " AND job_title LIKE ?";
                $params[] = '%' . $filters['job_title'] . '%';
            }
            
            if (!empty($filters['status'])) {
                $sql .= " AND status = ?";
                $params[] = $filters['status'];
            }
            
            if (!empty($filters['city'])) {
                $sql .= " AND city LIKE ?";
                $params[] = '%' . $filters['city'] . '%';
            }
            
            if (!empty($filters['state'])) {
                $sql .= " AND state = ?";
                $params[] = $filters['state'];
            }
            
            if (!empty($filters['department'])) {
                $sql .= " AND department LIKE ?";
                $params[] = '%' . $filters['department'] . '%';
            }
            
            if (!empty($filters['work_location'])) {
                $sql .= " AND work_location LIKE ?";
                $params[] = '%' . $filters['work_location'] . '%';
            }
            
            $sql .= " ORDER BY full_name";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            throw new Exception('Erro ao buscar colaboradores: ' . $e->getMessage());
        }
    }
    
    /**
     * Buscar colaboradores por proximidade geográfica
     */
    public function findByLocation($latitude, $longitude, $radius = 10) {
        try {
            $sql = "SELECT *, 
                    (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * 
                    cos(radians(longitude) - radians(?)) + sin(radians(?)) * 
                    sin(radians(latitude)))) AS distance 
                    FROM employees 
                    WHERE latitude IS NOT NULL AND longitude IS NOT NULL 
                    HAVING distance < ? 
                    ORDER BY distance";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([$latitude, $longitude, $latitude, $radius]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            throw new Exception('Erro ao buscar por localização: ' . $e->getMessage());
        }
    }
    
    /**
     * Obter estatísticas por localização
     */
    public function getLocationStats() {
        try {
            $sql = "SELECT 
                    city, 
                    state, 
                    department,
                    work_location,
                    COUNT(*) as total,
                    SUM(CASE WHEN status = 'ativo' THEN 1 ELSE 0 END) as ativos
                    FROM employees 
                    GROUP BY city, state, department, work_location
                    ORDER BY total DESC";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            throw new Exception('Erro ao obter estatísticas: ' . $e->getMessage());
        }
    }
    
    /**
     * Exclusão em lote
     */
    public function deleteMultiple($ids) {
        try {
            if (empty($ids)) {
                return false;
            }
            
            $placeholders = str_repeat('?,', count($ids) - 1) . '?';
            $sql = "DELETE FROM employees WHERE id IN ($placeholders)";
            
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($ids);
        } catch (PDOException $e) {
            throw new Exception('Erro ao deletar colaboradores: ' . $e->getMessage());
        }
    }
    
    /**
     * Atualização em lote
     */
    public function updateMultiple($ids, $data) {
        try {
            if (empty($ids) || empty($data)) {
                return false;
            }
            
            $setParts = [];
            $params = [];
            
            foreach ($data as $field => $value) {
                $setParts[] = "$field = ?";
                $params[] = $value;
            }
            
            $placeholders = str_repeat('?,', count($ids) - 1) . '?';
            $sql = "UPDATE employees SET " . implode(', ', $setParts) . 
                   ", updated_at = NOW() WHERE id IN ($placeholders)";
            
            $params = array_merge($params, $ids);
            
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($params);
        } catch (PDOException $e) {
            throw new Exception('Erro ao atualizar colaboradores: ' . $e->getMessage());
        }
    }
    
    /**
     * Método para compatibilidade com código existente
     */
    public function listEmployees() {
        return $this->getAll();
    }
}