<?php
class Documento {
    private $pdo;

    // Tipos de arquivo permitidos por categoria
    private $tiposPermitidos = [
        'documento' => ['pdf', 'doc', 'docx', 'txt', 'odt'],
        'imagem' => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'],
        'video' => ['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm']
    ];

    // Tamanho máximo por tipo (em bytes)
    private $tamanhoMaximo = [
        'documento' => 10485760, // 10MB
        'imagem' => 5242880,     // 5MB
        'video' => 104857600     // 100MB
    ];

    public function __construct() {
        try {
            $this->pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch (PDOException $e) { die("Erro na conexão com o banco de dados: " . $e->getMessage()); }
    }

    public function create($data) {
        // Versão simplificada até as colunas serem criadas no banco
        $sql = "INSERT INTO documentos (id_usuario, titulo, nome_arquivo, caminho_arquivo, tipo_arquivo, tamanho_arquivo, categoria) VALUES (:id_usuario, :titulo, :nome_arquivo, :caminho_arquivo, :tipo_arquivo, :tamanho_arquivo, :categoria)";
        $stmt = $this->pdo->prepare($sql);
        
        // Filtrar apenas os dados que existem na tabela atual
        $filteredData = [
            ':id_usuario' => $data['id_usuario'],
            ':titulo' => $data['titulo'],
            ':nome_arquivo' => $data['nome_arquivo'],
            ':caminho_arquivo' => $data['caminho_arquivo'],
            ':tipo_arquivo' => $data['tipo_arquivo'],
            ':tamanho_arquivo' => $data['tamanho_arquivo'],
            ':categoria' => $data['categoria'] ?? 'Documento'
        ];
        
        return $stmt->execute($filteredData);
    }

    public function find($categoria = null, $tipo_midia = null) {
        $sql = "SELECT d.*, u.nome as nome_usuario FROM documentos d JOIN usuarios u ON d.id_usuario = u.id WHERE 1=1";
        $params = [];
        
        if ($categoria && $categoria != '') {
            $sql .= " AND d.categoria = :categoria";
            $params['categoria'] = $categoria;
        }
        
        // Temporariamente comentado até a coluna ser criada no banco
         /*
         if ($tipo_midia && $tipo_midia != '') {
             $sql .= " AND d.tipo_midia = :tipo_midia";
             $params['tipo_midia'] = $tipo_midia;
         }
         */
        
        $sql .= " ORDER BY d.data_upload DESC";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Valida se o arquivo é permitido para o tipo de mídia
     */
    public function validarArquivo($arquivo, $tipo_midia) {
        $extensao = strtolower(pathinfo($arquivo['name'], PATHINFO_EXTENSION));
        
        // Verifica se a extensão é permitida
        if (!in_array($extensao, $this->tiposPermitidos[$tipo_midia])) {
            return ['valido' => false, 'erro' => 'Tipo de arquivo não permitido para ' . $tipo_midia];
        }
        
        // Verifica o tamanho
        if ($arquivo['size'] > $this->tamanhoMaximo[$tipo_midia]) {
            $tamanhoMB = $this->tamanhoMaximo[$tipo_midia] / 1048576;
            return ['valido' => false, 'erro' => 'Arquivo muito grande. Máximo: ' . $tamanhoMB . 'MB'];
        }
        
        return ['valido' => true];
    }

    /**
     * Determina o tipo de mídia baseado na extensão do arquivo
     */
    public function determinarTipoMidia($nomeArquivo) {
        $extensao = strtolower(pathinfo($nomeArquivo, PATHINFO_EXTENSION));
        
        foreach ($this->tiposPermitidos as $tipo => $extensoes) {
            if (in_array($extensao, $extensoes)) {
                return $tipo;
            }
        }
        
        return 'documento'; // padrão
    }

    /**
     * Obtém informações de imagem (largura e altura)
     */
    public function obterInfoImagem($caminhoArquivo) {
        if (function_exists('getimagesize')) {
            $info = getimagesize($caminhoArquivo);
            if ($info) {
                return ['largura' => $info[0], 'altura' => $info[1]];
            }
        }
        return ['largura' => null, 'altura' => null];
    }

    /**
     * Busca documentos por título (versão simplificada)
     */
    public function buscar($termo, $tipo_midia = null) {
        // Versão simplificada usando apenas colunas existentes
        $sql = "SELECT d.*, u.nome as nome_usuario FROM documentos d 
                JOIN usuarios u ON d.id_usuario = u.id 
                WHERE d.titulo LIKE :termo";
        
        $params = ['termo' => '%' . $termo . '%'];
        
        // Comentado até a coluna tipo_midia ser criada
        /*
        if ($tipo_midia && $tipo_midia != '') {
            $sql .= " AND d.tipo_midia = :tipo_midia";
            $params['tipo_midia'] = $tipo_midia;
        }
        */
        
        $sql .= " ORDER BY d.data_upload DESC";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Obtém estatísticas dos documentos
     */
    public function obterEstatisticas() {
        $sql = "SELECT 
                    tipo_midia,
                    categoria,
                    COUNT(*) as total,
                    SUM(tamanho_arquivo) as tamanho_total
                FROM documentos 
                GROUP BY tipo_midia, categoria
                ORDER BY tipo_midia, categoria";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}